@extends('layouts.admin')

@section('title', 'Error Logs')

@section('content')
<div class="dashboard-header">
    <h1>Error Logs</h1>
    <p>Monitor and manage application errors</p>
</div>

{{-- Statistics Cards --}}
<div class="dashboard-stats">
    <div class="stat-card">
        <div class="stat-card-inner">
            <h3>Total Errors</h3>
            <div class="stat-value">{{ number_format($stats['total']) }}</div>
            <div class="stat-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
        </div>
    </div>

    <div class="stat-card">
        <div class="stat-card-inner">
            <h3>Unresolved</h3>
            <div class="stat-value" style="color: var(--danger);">{{ number_format($stats['unresolved']) }}</div>
            <div class="stat-icon">
                <i class="fas fa-times-circle"></i>
            </div>
        </div>
    </div>

    <div class="stat-card">
        <div class="stat-card-inner">
            <h3>Critical</h3>
            <div class="stat-value" style="color: var(--danger);">{{ number_format($stats['critical']) }}</div>
            <div class="stat-icon">
                <i class="fas fa-bomb"></i>
            </div>
        </div>
    </div>

    <div class="stat-card">
        <div class="stat-card-inner">
            <h3>Today</h3>
            <div class="stat-value">{{ number_format($stats['today']) }}</div>
            <div class="stat-icon">
                <i class="fas fa-calendar-day"></i>
            </div>
        </div>
    </div>
</div>

{{-- Filters --}}
<div class="dashboard-card mb-4">
    <div class="dashboard-card-header">
        <h3>Filters</h3>
    </div>
    <div class="dashboard-card-body">
        <form method="GET" action="{{ route('admin.errors.index') }}" class="form-inline">
            <div class="row">
                <div class="col-md-3">
                    <div class="form-group">
                        <label>Search</label>
                        <input type="text" name="search" class="form-control" placeholder="Search errors..." value="{{ request('search') }}">
                    </div>
                </div>
                
                <div class="col-md-2">
                    <div class="form-group">
                        <label>Level</label>
                        <select name="level" class="form-control">
                            <option value="">All Levels</option>
                            <option value="critical" {{ request('level') === 'critical' ? 'selected' : '' }}>Critical</option>
                            <option value="error" {{ request('level') === 'error' ? 'selected' : '' }}>Error</option>
                            <option value="warning" {{ request('level') === 'warning' ? 'selected' : '' }}>Warning</option>
                        </select>
                    </div>
                </div>
                
                <div class="col-md-2">
                    <div class="form-group">
                        <label>Status</label>
                        <select name="status" class="form-control">
                            <option value="">All</option>
                            <option value="unresolved" {{ request('status') === 'unresolved' ? 'selected' : '' }}>Unresolved</option>
                            <option value="resolved" {{ request('status') === 'resolved' ? 'selected' : '' }}>Resolved</option>
                        </select>
                    </div>
                </div>
                
                <div class="col-md-2">
                    <div class="form-group">
                        <label>From</label>
                        <input type="date" name="date_from" class="form-control" value="{{ request('date_from') }}">
                    </div>
                </div>
                
                <div class="col-md-2">
                    <div class="form-group">
                        <label>To</label>
                        <input type="date" name="date_to" class="form-control" value="{{ request('date_to') }}">
                    </div>
                </div>
                
                <div class="col-md-1 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-search"></i>
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

{{-- Actions Bar --}}
<div class="dashboard-card mb-4">
    <div class="dashboard-card-body">
        <div class="d-flex justify-content-between align-items-center">
            <div class="bulk-actions" style="display: none;">
                <span class="selected-count">0 errors selected</span>
                <div class="btn-group ml-3">
                    <button type="button" class="btn btn-sm btn-success bulk-resolve">
                        <i class="fas fa-check"></i> Resolve
                    </button>
                    <button type="button" class="btn btn-sm btn-primary bulk-send">
                        <i class="fas fa-paper-plane"></i> Send to Support
                    </button>
                    <button type="button" class="btn btn-sm btn-info bulk-download">
                        <i class="fas fa-download"></i> Download
                    </button>
                    <button type="button" class="btn btn-sm btn-danger bulk-delete">
                        <i class="fas fa-trash"></i> Delete
                    </button>
                </div>
            </div>
            
            <div class="main-actions">
                <button type="button" class="btn btn-sm btn-warning" onclick="clearResolved()">
                    <i class="fas fa-broom"></i> Clear Resolved
                </button>
                <button type="button" class="btn btn-sm btn-secondary" onclick="clearOld()">
                    <i class="fas fa-clock"></i> Clear Old (30+ days)
                </button>
                <a href="{{ route('admin.errors.index') }}" class="btn btn-sm btn-outline-primary">
                    <i class="fas fa-redo"></i> Refresh
                </a>
            </div>
        </div>
    </div>
</div>

{{-- Error Logs Table --}}
<div class="dashboard-card">
    <div class="dashboard-card-body p-0">
        @if($errors->count() > 0)
        <div class="table-responsive">
            <table class="table">
                <thead>
                    <tr>
                        <th width="30">
                            <input type="checkbox" id="select-all">
                        </th>
                        <th>Level</th>
                        <th>Type</th>
                        <th>Message</th>
                        <th>File</th>
                        <th>User</th>
                        <th>Date</th>
                        <th>Status</th>
                        <th width="200">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($errors as $error)
                    <tr>
                        <td>
                            <input type="checkbox" class="error-select" value="{{ $error->id }}">
                        </td>
                        <td>
                            <span class="status-badge status-{{ $error->level_color }}">
                                {{ strtoupper($error->level) }}
                            </span>
                        </td>
                        <td>
                            <code style="font-size: 0.8rem;">{{ class_basename($error->type) }}</code>
                        </td>
                        <td>
                            <div style="max-width: 300px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;" title="{{ $error->message }}">
                                {{ $error->short_message }}
                            </div>
                        </td>
                        <td>
                            <small>{{ $error->file_location }}</small>
                        </td>
                        <td>
                            @if($error->user)
                                <a href="{{ route('admin.users.show', $error->user) }}">{{ $error->user->name }}</a>
                            @else
                                <span class="text-muted">Guest</span>
                            @endif
                        </td>
                        <td>
                            <small>{{ $error->created_at->format('M d, Y H:i') }}</small>
                        </td>
                        <td>
                            @if($error->is_resolved)
                                <span class="status-badge status-completed">Resolved</span>
                            @else
                                <span class="status-badge status-pending">Unresolved</span>
                            @endif
                            @if($error->is_sent)
                                <i class="fas fa-paper-plane text-info ml-1" title="Sent to support"></i>
                            @endif
                        </td>
                        <td>
                            <div class="user-actions">
                                <a href="{{ route('admin.errors.show', $error) }}" class="btn-icon btn-info" title="View Details">
                                    <i class="fas fa-eye"></i>
                                </a>
                                
                                @if(!$error->is_resolved)
                                <button type="button" class="btn-icon btn-success resolve-error" data-id="{{ $error->id }}" title="Mark as Resolved">
                                    <i class="fas fa-check"></i>
                                </button>
                                @endif
                                
                                <button type="button" class="btn-icon btn-primary send-error" data-id="{{ $error->id }}" title="Send to Support">
                                    <i class="fas fa-paper-plane"></i>
                                </button>
                                
                                <a href="{{ route('admin.errors.download', $error) }}" class="btn-icon btn-warning" title="Download">
                                    <i class="fas fa-download"></i>
                                </a>
                                
                                <button type="button" class="btn-icon btn-danger delete-error" data-id="{{ $error->id }}" title="Delete">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
        
        <div class="p-3">
            {{ $errors->links() }}
        </div>
        @else
        <div class="empty-state">
            <i class="fas fa-check-circle"></i>
            <h4>No Errors Found</h4>
            <p>Great! Your application is running smoothly with no errors logged.</p>
        </div>
        @endif
    </div>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const selectAll = document.getElementById('select-all');
    const errorSelects = document.querySelectorAll('.error-select');
    const bulkActions = document.querySelector('.bulk-actions');
    const mainActions = document.querySelector('.main-actions');
    const selectedCount = document.querySelector('.selected-count');

    // Select all checkbox
    if (selectAll) {
        selectAll.addEventListener('change', function() {
            errorSelects.forEach(cb => cb.checked = this.checked);
            updateBulkActions();
        });
    }

    // Individual checkboxes
    errorSelects.forEach(cb => {
        cb.addEventListener('change', updateBulkActions);
    });

    function updateBulkActions() {
        const selected = Array.from(errorSelects).filter(cb => cb.checked);
        if (selected.length > 0) {
            bulkActions.style.display = 'flex';
            mainActions.style.display = 'none';
            selectedCount.textContent = `${selected.length} error${selected.length > 1 ? 's' : ''} selected`;
        } else {
            bulkActions.style.display = 'none';
            mainActions.style.display = 'flex';
        }
    }

    // Bulk actions
    document.querySelector('.bulk-resolve')?.addEventListener('click', () => bulkAction('resolve'));
    document.querySelector('.bulk-send')?.addEventListener('click', () => bulkAction('send'));
    document.querySelector('.bulk-download')?.addEventListener('click', () => bulkAction('download'));
    document.querySelector('.bulk-delete')?.addEventListener('click', () => {
        if (confirm('Are you sure you want to delete the selected errors?')) {
            bulkAction('delete');
        }
    });

    function bulkAction(action) {
        const selected = Array.from(errorSelects).filter(cb => cb.checked).map(cb => cb.value);
        if (selected.length === 0) return;

        const routes = {
            resolve: '{{ route("admin.errors.bulk-resolve") }}',
            send: '{{ route("admin.errors.bulk-send") }}',
            download: '{{ route("admin.errors.bulk-download") }}',
            delete: '{{ route("admin.errors.bulk-delete") }}'
        };

        if (action === 'download') {
            window.location.href = routes[action] + '?ids=' + selected.join(',');
            return;
        }

        fetch(routes[action], {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify({ ids: selected })
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                showNotification(data.message, 'success');
                setTimeout(() => location.reload(), 1500);
            } else {
                showNotification(data.message, 'error');
            }
        });
    }

    // Individual actions
    document.querySelectorAll('.resolve-error').forEach(btn => {
        btn.addEventListener('click', function() {
            const id = this.dataset.id;
            fetch(`/admin/errors/${id}/resolve`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    showNotification(data.message, 'success');
                    setTimeout(() => location.reload(), 1000);
                }
            });
        });
    });

    document.querySelectorAll('.send-error').forEach(btn => {
        btn.addEventListener('click', function() {
            const id = this.dataset.id;
            const originalHtml = this.innerHTML;
            this.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            this.disabled = true;

            fetch(`/admin/errors/${id}/send`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(r => r.json())
            .then(data => {
                showNotification(data.message, data.success ? 'success' : 'error');
                if (data.success) {
                    setTimeout(() => location.reload(), 1500);
                }
            })
            .finally(() => {
                this.innerHTML = originalHtml;
                this.disabled = false;
            });
        });
    });

    document.querySelectorAll('.delete-error').forEach(btn => {
        btn.addEventListener('click', function() {
            if (!confirm('Are you sure you want to delete this error log?')) return;
            
            const id = this.dataset.id;
            fetch(`/admin/errors/${id}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    showNotification(data.message, 'success');
                    setTimeout(() => location.reload(), 1000);
                }
            });
        });
    });

    function showNotification(message, type) {
        if (window.adminPanel && window.adminPanel.showNotification) {
            window.adminPanel.showNotification(message, type);
        } else {
            alert(message);
        }
    }
});

function clearResolved() {
    if (!confirm('Are you sure you want to clear all resolved errors?')) return;

    fetch('{{ route("admin.errors.clear-resolved") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(r => r.json())
    .then(data => {
        alert(data.message);
        location.reload();
    });
}

function clearOld() {
    if (!confirm('Are you sure you want to clear all errors older than 30 days?')) return;

    fetch('{{ route("admin.errors.clear-old") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(r => r.json())
    .then(data => {
        alert(data.message);
        location.reload();
    });
}
</script>
@endpush